const express = require('express');
const moment = require('moment-timezone');
const Class = require('../../models/Class');
const Student = require('../../models/student/Student');
const Teacher = require('../../models/teacher/Teacher');
const { adminAuth, teacherAuth, studentAuth } = require('../../middleware/auth');
const { getCountryTimezone } = require('../../utils/syllabusHelper');

const router = express.Router();

router.post('/schedule', adminAuth, async (req, res) => {
  try {
    const {
      studentId,
      teacherId,
      subject,
      scheduledDate,
      duration,
      timezone,
      recurringSchedule
    } = req.body;

    const student = await Student.findById(studentId);
    const teacher = await Teacher.findById(teacherId);

    if (!student || !teacher) {
      return res.status(404).json({ message: 'Student or Teacher not found' });
    }

    if (!student.subjects.includes(subject) || !teacher.subjects.includes(subject)) {
      return res.status(400).json({ message: 'Subject mismatch between student and teacher' });
    }

    const classes = [];
    const startDate = moment.tz(scheduledDate, timezone);

    if (recurringSchedule?.isRecurring) {
      const { frequency, weeklyDays, customDates, endDate } = recurringSchedule;
      let currentDate = startDate.clone();
      const finalDate = moment.tz(endDate, timezone);

      while (currentDate.isSameOrBefore(finalDate)) {
        if (frequency === 'daily') {
          classes.push({
            studentId,
            teacherId,
            subject,
            scheduledDate: currentDate.toDate(),
            duration,
            timezone,
            recurringSchedule,
            meetingLink: student.googleMeetLink
          });
          currentDate.add(1, 'day');
        } else if (frequency === 'weekly') {
          const dayName = currentDate.format('dddd');
          if (weeklyDays.includes(dayName)) {
            classes.push({
              studentId,
              teacherId,
              subject,
              scheduledDate: currentDate.toDate(),
              duration,
              timezone,
              recurringSchedule,
              meetingLink: student.googleMeetLink
            });
          }
          currentDate.add(1, 'day');
        } else if (frequency === 'custom' && customDates) {
          customDates.forEach(customDate => {
            const customMoment = moment.tz(customDate, timezone);
            if (customMoment.isSameOrBefore(finalDate)) {
              classes.push({
                studentId,
                teacherId,
                subject,
                scheduledDate: customMoment.toDate(),
                duration,
                timezone,
                recurringSchedule,
                meetingLink: student.googleMeetLink
              });
            }
          });
          break;
        }
      }
    } else {
      classes.push({
        studentId,
        teacherId,
        subject,
        scheduledDate: startDate.toDate(),
        duration,
        timezone,
        meetingLink: student.googleMeetLink
      });
    }

    const savedClasses = await Class.insertMany(classes);

    // Update student's assigned teachers if not already assigned for this subject
    const existingAssignment = student.assignedTeachers.find(
      assignment => assignment.teacherId.toString() === teacherId.toString() && assignment.subject === subject
    );

    if (!existingAssignment) {
      student.assignedTeachers.push({
        teacherId: teacherId,
        subject: subject
      });
      await student.save();
    }

    res.status(201).json({ 
      message: 'Classes scheduled successfully', 
      classes: savedClasses,
      count: savedClasses.length 
    });
  } catch (error) {
    console.error('Schedule class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/:id/reschedule', adminAuth, async (req, res) => {
  try {
    const { newDate, timezone, reason } = req.body;
    
    // Validation
    if (!newDate) {
      return res.status(400).json({ message: 'New date is required' });
    }
    
    if (!timezone) {
      return res.status(400).json({ message: 'Timezone is required' });
    }
    
    console.log('Reschedule request received:', { classId: req.params.id, newDate, timezone, reason });
    
    const classToUpdate = await Class.findById(req.params.id);

    if (!classToUpdate) {
      return res.status(404).json({ message: 'Class not found' });
    }

    // Parse the new date and extract time components
    const newDateTime = moment.tz(newDate, timezone);
    const newStartTime = newDateTime.format('HH:mm');
    const newEndTime = newDateTime.clone().add(classToUpdate.duration, 'minutes').format('HH:mm');

    // Store original date for logging
    const originalDate = classToUpdate.scheduledDate;
    
    // Update the existing class with new date/time
    classToUpdate.scheduledDate = newDateTime.toDate();
    classToUpdate.startTime = newStartTime;
    classToUpdate.endTime = newEndTime;
    classToUpdate.timezone = timezone;
    classToUpdate.status = 'scheduled'; // Reset status to scheduled
    
    // Add reschedule tracking (optional - to keep history)
    if (reason) {
      classToUpdate.rescheduleReason = reason;
    }
    
    await classToUpdate.save();
    console.log('Class rescheduled successfully:', {
      classId: classToUpdate._id,
      from: originalDate,
      to: classToUpdate.scheduledDate,
      timezone
    });

    res.json({ 
      message: 'Class rescheduled successfully', 
      class: classToUpdate,
      rescheduled: {
        from: originalDate,
        to: classToUpdate.scheduledDate,
        timezone
      }
    });
    console.log('Reschedule completed successfully');
  } catch (error) {
    console.error('Reschedule class error:', error);
    console.error('Error details:', error.message);
    console.error('Stack trace:', error.stack);
    res.status(500).json({ 
      message: 'Failed to reschedule class: ' + error.message,
      error: error.message 
    });
  }
});

router.get('/calendar', async (req, res) => {
  try {
    const { view = 'month', date = new Date(), timezone = 'UTC' } = req.query;
    
    let startDate, endDate;
    const currentDate = moment.tz(date, timezone);

    switch (view) {
      case 'day':
        startDate = currentDate.clone().startOf('day');
        endDate = currentDate.clone().endOf('day');
        break;
      case 'week':
        startDate = currentDate.clone().startOf('week');
        endDate = currentDate.clone().endOf('week');
        break;
      case 'month':
        startDate = currentDate.clone().startOf('month');
        endDate = currentDate.clone().endOf('month');
        break;
      default:
        startDate = currentDate.clone().startOf('month');
        endDate = currentDate.clone().endOf('month');
    }

    const classes = await Class.find({
      scheduledDate: {
        $gte: startDate.toDate(),
        $lte: endDate.toDate()
      },
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
    .populate('studentId', 'studentName')
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const formattedClasses = classes.map(cls => ({
      ...cls.toObject(),
      formattedDate: moment.tz(cls.scheduledDate, timezone).format('YYYY-MM-DD HH:mm'),
      localTime: moment.tz(cls.scheduledDate, timezone).format('HH:mm')
    }));

    res.json({
      classes: formattedClasses,
      view,
      startDate: startDate.format('YYYY-MM-DD'),
      endDate: endDate.format('YYYY-MM-DD')
    });
  } catch (error) {
    console.error('Get calendar error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/student/:studentId', studentAuth, async (req, res) => {
  try {
    const classes = await Class.find({
      studentId: req.params.studentId,
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: -1 });

    res.json({ classes });
  } catch (error) {
    console.error('Get student classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/teacher/:teacherId', teacherAuth, async (req, res) => {
  try {
    const classes = await Class.find({
      teacherId: req.params.teacherId,
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
    .populate('studentId', 'studentName')
    .sort({ scheduledDate: -1 });

    res.json({ classes });
  } catch (error) {
    console.error('Get teacher classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/:id/attendance', teacherAuth, async (req, res) => {
  try {
    const { studentAttended, teacherAttended, classNotes, homeworkAssigned } = req.body;
    
    const updatedClass = await Class.findByIdAndUpdate(
      req.params.id,
      {
        'attendanceStatus.studentAttended': studentAttended,
        'attendanceStatus.teacherAttended': teacherAttended,
        classNotes,
        homeworkAssigned,
        status: 'completed'
      },
      { new: true }
    );

    if (!updatedClass) {
      return res.status(404).json({ message: 'Class not found' });
    }

    res.json({ message: 'Attendance updated successfully', class: updatedClass });
  } catch (error) {
    console.error('Update attendance error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/:id', adminAuth, async (req, res) => {
  try {
    const deletedClass = await Class.findByIdAndUpdate(
      req.params.id,
      { status: 'cancelled' },
      { new: true }
    );

    if (!deletedClass) {
      return res.status(404).json({ message: 'Class not found' });
    }

    res.json({ message: 'Class cancelled successfully' });
  } catch (error) {
    console.error('Cancel class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;